/* 
DATE: August 24, 2009
PURPOSE: Plot multiple versions of lifecycle values
TAKES IN: RAND+ADAMS+HRS_Cognitive_Measures.dta
PUTS OUT: Lifecycle Graphs
*/

// SETTINGS:
local log_p = "true"
local latex_p = "trued"
local reset_p = "true"

// RESET:
if "`reset_p'" == "true" {
	clear
	set mem 100m
	// set maxvar 7000
}

// NAMES:
local dir "/Users/Bowen/Documents/Laibson/HRS/Cognition Files/"
local lfilename = "Naive-Control Graphs Log" // log name
*local tfilename "TICS Graphs"
local tfilename "HRS Naive - Control Graphs (Method 2, Obs from 980) CLEAN" // tex name
local usefile "RAND+ADAMS+HRS_Cognitive_Measures"

// PRELIMINARIES:
set more off
local time_now = substr(c(current_date),4,3) + substr(c(current_date),1,2) + " | " + subinstr(substr(c(current_time),1,5),":","-",1)
timer clear
timer on 47

// LOG:
if "`log_p'" == "true" {
	capture log close
	set logtype text
	log using "`dir'Logs/`lfilename' (`time_now').txt", replace
}

// LATEX:
if "`latex_p'" == "true" {
	capture: erase "`dir'`tfilename'.tex"
	capture file close tfile
	file open tfile using "`dir'`tfilename'.tex", write replace
	#delimit ;
	file write tfile 
		"\documentclass[11pt]{article}" _n
		"\usepackage[pdftex]{graphicx}" _n(2)
		"\begin{document}" _n(2);	
	#delimit cr
}

//\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
//\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\ 

// more settings
local size "full"
local min_vars_cutoff 1000
local limit = 5.0
*local cogmeasures "mytics tics1_ tics2_ tics3_ tics4_ tics5_ tics7_ tics8_ tics9_ tics10_"
local cogmeasures "cgdisease lottery"
local cogstar ""
foreach m in `cogmeasures' {
	local cogstar "`cogstar' `m'*"
}

// list variable labels

local summary_label "Summary"
local immwords_label "Immediate Word Recall"
local delwords_label "Delayed Word Recall"
local serial7s_label "Serial 7s"
local tics_label "TICS"
local mytics_label "TICS"
local cgdisease_label "Chance of Disease"
local lottery_label "Split Lottery"
local interest_label "Interest on 200 Dollars"

local tics1__label "Current Month"
local tics2__label "Current Day"
local tics3__label "NCurrent Year"
local tics4__label "Current Day-Week"
local tics5__label "Countdown from 20"
local tics7__label "Cut-paper"
local tics8__label "Cactus"
local tics9__label "President"
local tics10__label "Vice President"

// list variable summaries

#delimit ;
local summary_sum "Overall summary score is the sum total of scores from 
	the immediate word recall, delayed word recall, tics, and serial sevens tests.";
local immwords_sum "The interviewer read a list of 10 nouns 
	(e.g., lake, car, army, etc.) to the respondent, and asked the respondent 
	to recall as many words as possible from the list in any order. [Score =
	number of words recalled.]";
local delwords_sum "The interviewer returned to the word list read off to the subject
	earlier, and again asked the subject to recall as many words as possible. [Score =
	number of words recalled.]";
local serial7s_sum "The interviewer asked the respondent to subtract 7 from 100, 
	and continue subtracting 7 from each subsequent number for a total of 5 trials. 
	It was up to the respondent to remember the value from the prior subtraction, such 
	that the interviewer did not repeat the difference said by the respondent 
	after each trial. [Score = number of trials answered correctly.]";
local tics_sum "Sum of 0 or 1 scores from the following questions: What is the current
	year? Month? Day? Day of the week? What do you usually use to cut paper? 
	What do you call the kind of prickly plant that grows in the desert? 
	Who is the current president? Vice president?, plus two chances to count 
	backwards from twenty to ten. [Score = sum of binary scores. Ranges 0 - 10.]";
local mytics_sum "`tics_sum'";
local cgdisease_sum "Question: If the chance of getting a disease is 10 
	percent, how many people out of 1,000 would be expected to get the disease? 
	[Score = 1(answer = 100)]";
local lottery_sum "Question: If 5 people all have the winning numbers in the 
	lottery and the prize is two million dollars, how much will each of them get?
	[Score = 1(answer = 400,000)]";
local interest_sum "Question: Let's say you have 200 dollars in a savings account. 
	The account earns ten percent interest per year. How much would you have in the 
	account at the end of two years? [Score = 1(answer = 242)]";
#delimit cr

local tics1__sum ""
local tics2__sum ""
local tics3__sum ""
local tics4__sum ""
local tics5__sum ""
local tics7__sum ""
local tics8__sum ""
local tics9__sum ""
local tics10__sum ""

// make report:

foreach measure in `cogmeasures' {
	
	clear
	use "`dir'`usefile'"
	
	// manage dataset size
	if "`size'" != "full" {
		sample `size', count	
	}
	else {
		local size = _N
	}
	
	// create age variable

	forvalues n = 1 / 8 {
		gen rintage`n' = (r`n'iwmid - rabdate) / 365.25	
	}
	
	keep hhidpn rintage* `cogstar'

	// generate age buckets
	
	foreach stub in "d" "naive" "control" {
		forvalues n = 20 / 110 {
			quietly gen `stub'`n' = .
		}
	}
	
	// fill age variables

	local N = _N
	
	forvalues n = 1 / `N' {
		forvalues x = 1 / 8 {
			
			local mx = `measure'`x' in `n'
			local ax = rintage`x' in `n'
			local ax_i = round(`ax')
			
			// setting naive values
			if (`mx' != .) & (`ax' != .) {
				
				quietly replace naive`ax_i' = `mx' in `n'
			
				local y = `x' + 1
				local loop_again = "true"

				if `y' > 8 {
					local loop_again = "false"
				}

				// principal loop:
				while "`loop_again'" == "true" {

					if `y' > 8 {
						local loop_again = "false"
					}
					else {
						
						local my = `measure'`y' in `n'
						local ay = rintage`y' in `n'
						local ay_i = round(`ay')
						
						local adiff = `ay' - `ax'

						// setting slope values
						if (`my' != .) & (`ay' != .) & (`adiff' < `limit') {

							local ax_fl = floor(`ax')
							local ay_fl = floor(`ay')

							local mdiff = `my' - `mx'

							local start = `ax_fl' + 1
							local finish = `ay_fl'
						
							forvalues z = `start' / `finish' {
								local slope = `mdiff' / `adiff'
								quietly replace d`z' = `slope' in `n'
							}

							local loop_again = "false"

						}
						else {
							local y = `y' + 1
						}
					}

				}
			}	
		}
	}
	
	// export spreadsheet for checking work
	
	*keep hhidpn d* naive* control* `cogstar' rintage*
	*outsheet using "`dir'Spreadsheet for Checking.csv", replace comma
	
	keep hhidpn d* naive* control*
	
	// make N variables
	
	forvalues n = 20 / 110 {
		quietly egen num`n' = count(d`n')
	}
	
	// averaging and reshaping
	
	collapse (mean) d* naive* control* num* hhidpn
	reshape long d naive control num, i(hhidpn) j(age)
	drop hhidpn
	
	// setting t_0
	
	local t_0 = 0
	
	local max = _N
	forvalues n = 1 / `max' {
		if `t_0' == 0 {
			local temp = num in `n'
			if `temp' > `min_vars_cutoff' {
				local t_0 = `n'
				break
			}
		}
	}
	
	/* forvalues n = 1 / `max' {
		local temp = age in `n'
		if `temp' == `t_0' {
			local t_0 = `n'
			break
		}
	} */
	
	if `t_0' == 0 {
		local `measure'_label "``measure'_label' [NOT ENOUGH OBSERVATIONS!]"
	}
	
	// creating control values
	
	replace control = naive in `t_0'
	
	local t_1 = `t_0' + 1
	forvalues n = `t_1' / `max' {
		local n_1 = `n' - 1
		local delta = d in `n_1'
		local before = control in `n_1'
		di "`n'"
		replace control = `before' + `delta' in `n'
	}
	
	// printing
	
	local age_t_0 = age in `t_0'
	local xstart = `age_t_0' - 1
	local xend = 91
	
	label variable naive "Naive"
	label variable control "Control"
	label variable num "Num. of Obs."
	
	graph twoway /// (scatter naive age, connect(direct)) ///
		(scatter control age, connect(direct)) ///
		if age > `xstart' & age < `xend', ///
	 	title("Progression of ``measure'_label' Score") ///
		ytitle("``measure'_label' Score") xtitle("Age")
		*(scatter d age, connect(direct) yaxis(2)) ///
		*(scatter num age, connect(direct) yaxis(3)) ///
		
	graph export "`dir'Graphs/aaa``measure'_label' Naive - Control Graph.pdf", replace
	
	file write tfile ///
		"\clearpage" _n ///
		"\begin{figure}" _n ///
		"\begin{center}" _n ///
		"\includegraphics{`dir'Graphs/aaa``measure'_label' Naive - Control Graph.eps}" _n ///
		"\end{center}" _n ///
		"\b{``measure'_label'}: ``measure'_sum'" _n ///
		"\end{figure}" _n(2)  	
		
	save "`dir'Logs/`lfilename' Data (``measure'_label') (`time_now').dta", replace

}

////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////

if "`latex_p'" == "true" {
	file write tfile "\end{document}" _n
	file close tfile
	shell pdflatex "`dir'`tfilename'.tex" "`dir'../"

	***Erasing unnecessary files
	foreach extension in "aux" "log" {
		capture: erase "`dir'`tfilename'.`extension'"
	}
}

timer off 47
timer list 47
capture log close

exit